// Bateau class: Represents a ship with a certain number of health points
class Bateau {
    private pointsDeVie: number;
  
    constructor(longueur: number) {
      this.pointsDeVie = longueur;
    }
  
    estCoulé(): boolean {
      return this.pointsDeVie === 0;
    }
  
    êtreTouché(): void {
      if (!this.estCoulé()) {
        this.pointsDeVie--;
      }
    }
  
    getPointsDeVie(): number {
      return this.pointsDeVie;
    }
  }
  
  // Cellule class: Represents a cell on the board
  class Cellule {
    private _bateau: Bateau | null = null;
    private _touché = false;
  
    get bateauPrésent(): boolean {
      return this._bateau !== null;
    }
  
    set bateau(b: Bateau) {
      this._bateau = b;
    }
  
    estTouché(): boolean {
      return this._touché;
    }
  
    toucher(): string {
      if (this._touché) return "déjà touché";
      this._touché = true;
  
      if (this._bateau) {
        this._bateau.êtreTouché();
        return this._bateau.estCoulé() ? "coulé" : "touché";
      }
      return "dans l'eau";
    }
  }
  
  // Position class: Represents a coordinate on the grid
  class Position {
    constructor(public x: number, public y: number) {}
  
    égal(p: Position): boolean {
      return this.x === p.x && this.y === p.y;
    }
  }
  
  // Mer class: Represents the game board
  class Mer {
    private _plateau: Cellule[][];
  
    constructor(private _taille: number) {
      this._plateau = Array.from({ length: _taille }, () =>
        Array.from({ length: _taille }, () => new Cellule())
      );
    }
  
    ajouterBateau(b: Bateau, p: Position): void {
      if (this._plateau[p.x][p.y].bateauPrésent) {
        throw new Error("Bateau déjà présent à cette position");
      }
      this._plateau[p.x][p.y].bateau = b;
    }
  
    tirer(p: Position): string {
      if (p.x < 0 || p.x >= this._taille || p.y < 0 || p.y >= this._taille) {
        throw new Error("Position hors de la grille");
      }
      return this._plateau[p.x][p.y].toucher();
    }
  
    afficherGrille(): void {
      console.log("\nGrille:");
      for (let i = 0; i < this._taille; i++) {
        let ligne = "";
        for (let j = 0; j < this._taille; j++) {
          const cell = this._plateau[i][j];
          if (cell.bateauPrésent) {
            ligne += cell.estTouché() ? "X " : "B ";
          } else {
            ligne += cell.estTouché() ? "- " : "~ ";
          }
        }
        console.log(ligne);
      }
    }
  
    tousLesBateauxSontCoulés(): boolean {
      for (let i = 0; i < this._taille; i++) {
        for (let j = 0; j < this._taille; j++) {
          const cell = this._plateau[i][j];
          if (cell.bateauPrésent && !cell.estTouché()) {
            return false;
          }
        }
      }
      return true;
    }
  }
  
  // Main logic to run the game
  if (import.meta.main) {
    const mer = new Mer(5); // Create a 5x5 grid
  
    // Add ships to the grid
    const bateau1 = new Bateau(3);
    const bateau2 = new Bateau(2);
  
    try {
      mer.ajouterBateau(bateau1, new Position(1, 1)); // Add a ship at (1, 1)
      mer.ajouterBateau(bateau2, new Position(3, 3)); // Add a ship at (3, 3)
    } catch (error) {
      console.error("Erreur lors de l'ajout des bateaux:", error.message);
    }
  
    console.log("Bienvenue à Bataille Navale !");
    mer.afficherGrille();
  
    // Start the game loop
    while (!mer.tousLesBateauxSontCoulés()) {
      // Ask the player for coordinates
      const input = prompt("Entrez les coordonnées pour tirer (format: x,y): ");
      if (!input) {
        console.log("Fin de la partie.");
        break;
      }
  
      const [x, y] = input.split(",").map(Number);
      if (isNaN(x) || isNaN(y)) {
        console.log("Coordonnées invalides. Réessayez.");
        continue;
      }
  
      try {
        const result = mer.tirer(new Position(x, y));
        console.log(`Tir à (${x},${y}): ${result}`);
      } catch (error) {
        console.log("Erreur:", error.message);
      }
  
      mer.afficherGrille();
    }
  
    console.log("Félicitations ! Vous avez coulé tous les bateaux !");
  }
  